<?php
/**
 * Plugin Name: BSK Recordar
 * Description: Plugin básico para conectar WordPress con Bluesky y automatizar efemérides.
 * Version: 1.5.8-CLEAN-STABLE (ROLLBACK SIN EVENTO ESPECIAL)
 * Author: JRMora & AI Partner
 * Text Domain: bsk-recordar
 * Domain Path: /languages
 * * Este archivo implementa la publicación de efemérides espaciada, exclusión de posts por ID y herramienta de reinicio de CRON.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Evitar acceso directo
}

// =========================================================================================================
// !!! CORRECCIONES FINALES DE ESTABILIDAD Y HORA (MANTENIDAS) !!!
// =========================================================================================================

// SOLUCIÓN AL DESFASE DE 1 HORA DEL SERVIDOR
// Hacemos que WordPress programe los CRONs con 1 hora de adelanto (3600 segundos) para compensar el retraso.
add_filter( 'schedule_event', 'bsk_recordar_adjust_schedule_time' );
function bsk_recordar_adjust_schedule_time( $event ) {
    if ( $event->hook == 'bsk_recordar_timed_event' ) {
        // Adelantamos la programación 3600 segundos (1 hora)
        $event->timestamp = $event->timestamp - 3600; 
    }
    return $event;
}

// ----------------------------------------------------
// 1. GESTIÓN DEL CRON Y HELPER DE PARSEO
// ----------------------------------------------------

register_activation_hook( __FILE__, 'bsk_recordar_activate_cron' );
register_deactivation_hook( __FILE__, 'bsk_recordar_deactivate_cron' );

/**
 * Neteza robusta array-based de tots els esdeveniments programats per al hook.
 */
function bsk_recordar_clear_all_events() {
    $hook = 'bsk_recordar_timed_event';
    
    // 1. Limpieza estándar (por si hay sin argumentos)
    wp_clear_scheduled_hook( $hook ); 

    // 2. Limpieza robusta de todos los eventos (con y sin argumentos)
    $cron_jobs = function_exists('_get_cron_array') ? _get_cron_array() : get_option('cron');

    if (empty($cron_jobs) || !is_array($cron_jobs)) {
        return;
    }

    foreach ($cron_jobs as $timestamp => $job) {
        if (!is_numeric($timestamp)) continue; 
        if (isset($job[$hook]) && is_array($job[$hook])) {
            foreach ($job[$hook] as $key => $event) {
                $args = $event['args'] ?? [];
                wp_unschedule_event($timestamp, $hook, $args);
            }
        }
    }
}


/**
 * Parsea la configuración de horarios JSON en una estructura agrupada por tiempo.
 */
function bsk_recordar_parse_schedule( $general_config_array ) {
    if ( ! is_array( $general_config_array ) || empty ( $general_config_array ) ) {
        return [];
    }
    
    $groups = [];
    foreach ($general_config_array as $item) {
        if ( isset($item['year']) && isset($item['time']) ) {
            $year = intval($item['year']);
            $time = $item['time'];

            if ( $year > 0 && preg_match('/^\d{2}:\d{2}(:\d{2})?$/', $time) ) {
                if ( ! isset($groups[$time]) ) {
                    $groups[$time] = [];
                }
                $groups[$time][] = $year;
            }
        }
    }
    return $groups;
}

/**
 * Activa o re-programa todos los eventos de Cron dinámicamente.
 * SOLAMENTE PARA LA PROGRAMACIÓN GENERAL.
 */
function bsk_recordar_activate_cron() {
    $hook = 'bsk_recordar_timed_event';
    
    // Obtener las configuraciones guardadas (SOLO GENERAL)
    $general_config_json = get_option( 'bsk_schedule_groups_array', '[]' );
    $general_config_array = json_decode( $general_config_json, true );
    
    // Generar un array de todos los años por hora para el CRON
    $general_schedule_groups = bsk_recordar_parse_schedule( $general_config_array );
    
    $all_unique_times = array_keys($general_schedule_groups);

    // 1. Limpiar todos los eventos de este hook de forma robusta
    bsk_recordar_clear_all_events();

    // 2. Programar los nuevos eventos
    foreach ($all_unique_times as $time_string) {
        $years_array = $general_schedule_groups[$time_string] ?? [];
        // El hook recibe el array de años Y la hora esperada como argumento (HH:mm:00).
        wp_schedule_event( strtotime( $time_string ), 'daily', $hook, [$years_array, $time_string] );
    }
}

function bsk_recordar_deactivate_cron() {
    bsk_recordar_clear_all_events();
}

// ----------------------------------------------------
// 2. FUNCIÓN DE EJECUCIÓN DEL CRON
// ----------------------------------------------------

// ACEPTAR 3 ARGUMENTOS: years_array, expected_time_string, y is_manual_debug (aunque solo se usen 2)
add_action( 'bsk_recordar_timed_event', 'bsk_recordar_timed_publisher', 10, 3 ); 

/**
 * Función que se ejecuta en el Cron, SOLAMENTE maneja la programación general.
 */
function bsk_recordar_timed_publisher( $years_array, $expected_time_string = null, $is_manual_debug = false ) {
    
    $bsk_plugin = new BSK_Recordar();
    $posts_to_publish = [];

    // --- CÁLCULO DE HORA COMPENSADA ---
    $current_time_compensated = time() + 3600; 
    $log_prefix = "CRON ejecutado a {$expected_time_string} el " . date('Y-m-d H:i:s', $current_time_compensated) . " (Hora Compensada): ";
    
    // 1. VERIFICACIÓN DE HORA (BLOQUEO DE MISSED EVENTS)
    // Se mantiene esta lógica de seguridad de CRON
    if ( ! $is_manual_debug && defined( 'DOING_CRON' ) && DOING_CRON && $expected_time_string ) {
        $expected_timestamp = strtotime( date('Y-m-d', $current_time_compensated) . ' ' . $expected_time_string ); 
        
        // Si la hora actual es más de 5 minutos (300 segundos) después de la hora programada, ignoramos.
        if ( $current_time_compensated > ($expected_timestamp + 300) ) {
            $final_log = $log_prefix . "FALLO: Evento CRON perdido ('Missed Event'). La hora de ejecución es más de 5 minutos posterior a la programada.";
            update_option('bsk_last_cron_status', $final_log); // Usamos un nombre de log genérico
            return; 
        }
    }
    
    // 2. CHEQUEO DE PROGRAMACIÓN GENERAL DE EFEMÉRIDES (Solo si hay años configurados para esta hora)
    if ( ! empty( $years_array ) ) {
        $posts_to_publish = $bsk_plugin->bsk_recordar_query_posts( null, $years_array, [] );
    }
    
    // 3. LOG DE DIAGNÓSTICO FINAL (Simplificado)
    $final_log = $log_prefix;
    if (!empty($posts_to_publish)) {
        $count = count($posts_to_publish);
        $final_log .= "Éxito. Se encontraron {$count} posts para publicar.";
    } else {
         $final_log .= "No se encontraron posts para las efemérides de esta hora.";
    }
    update_option('bsk_last_cron_status', $final_log);


    // 4. PUBLICACIÓN
    if ( ! empty( $posts_to_publish ) ) {
        foreach ( $posts_to_publish as $post_data ) {
            
            $years_ago = $post_data['years_ago'];
            $plural = ($years_ago == 1) ? 'año' : 'años';
            $message = sprintf(
                'Publicado un día como hoy, hace %s %s: %s %s',
                $years_ago,
                $plural,
                $post_data['title'],
                $post_data['permalink'] 
            );
            
            $bsk_plugin->send_to_bluesky( $message, $post_data );
        }
    }
}

// ----------------------------------------------------
// 3. CLASE PRINCIPAL DEL PLUGIN
// ----------------------------------------------------

class BSK_Recordar {

    private $api_base = 'https://bsky.social/xrpc/';

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_post_bsk_test_publish', array( $this, 'handle_manual_publish' ) );
        add_action( 'admin_post_bsk_cron_reset', array( $this, 'handle_cron_reset' ) ); 
        
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        
        // Ejecutar bsk_recordar_activate_cron al guardar la opción de configuración general
        add_action( 'update_option_bsk_schedule_groups_array', 'bsk_recordar_activate_cron' );

        add_action( 'publish_post', array( $this, 'handle_new_post_publish' ), 10, 2 );
    }
    
    public function handle_cron_reset() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'bsk_cron_reset' ) ) {
            wp_die( 'Acción no permitida.' );
        }
        
        bsk_recordar_activate_cron(); 
        
        $redirect_url = add_query_arg( 'bsk_message', 'cron_reset', admin_url( 'admin.php?page=bsk-recordar' ) );
        wp_safe_redirect( $redirect_url );
        exit;
    }


    public function handle_manual_publish() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'bsk_manual_publish' ) ) {
            wp_die( 'Acción no permitida.' );
        }
        
        $test_date = isset($_POST['bsk_manual_publish_date']) ? sanitize_text_field($_POST['bsk_manual_publish_date']) : null;
        
        // Obtener la configuración general guardada
        $config_json = get_option( 'bsk_schedule_groups_array', '[]' );
        $config_array = json_decode( $config_json, true );

        $schedule_groups = bsk_recordar_parse_schedule( $config_array );
        
        $all_years = [];
        foreach ($schedule_groups as $years) {
            $all_years = array_merge($all_years, $years);
        }
        $years_to_check = array_unique($all_years);

        // En la prueba manual, solo buscamos la programación general de efemérides.
        $posts_to_publish = $this->bsk_recordar_query_posts( $test_date, $years_to_check, [] );
        
        $has_failed = false;
        $first_error_message = '';

        if ( ! empty( $posts_to_publish ) ) {
             foreach ( $posts_to_publish as $post_data ) {
                
                $years_ago = $post_data['years_ago'];
                $plural = ($years_ago == 1) ? 'año' : 'años';
                
                $message = sprintf(
                    'Publicado un día como hoy, hace %s %s: %s %s',
                    $years_ago,
                    $plural,
                    $post_data['title'],
                    $post_data['permalink'] 
                );
                
                $result = $this->send_to_bluesky( $message, $post_data );

                if ( is_wp_error( $result ) ) {
                    $has_failed = true;
                    if ( empty($first_error_message) ) {
                        $first_error_message = $result->get_error_message();
                    }
                }
            }
        }
        
        if ($has_failed) {
             $redirect_url = add_query_arg( 'bsk_error', urlencode($first_error_message), admin_url( 'admin.php?page=bsk-recordar' ) );
        } else {
             $redirect_url = add_query_arg( 'bsk_message', 'published', admin_url( 'admin.php?page=bsk-recordar' ) );
        }

        wp_safe_redirect( $redirect_url );
        exit;
    }

    public function handle_new_post_publish( $post_ID, $post ) {
        if ( get_option( 'bsk_autopublish_new', 'off' ) !== 'on' ) {
            return;
        }
        
        if ( $post->post_type !== 'post' || $post->post_status !== 'publish' || ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) ) {
            return;
        }
        
        // Excluir el post si está en la lista de IDs excluidas
        $excluded_ids_raw = get_option('bsk_exclude_ids', '');
        $excluded_ids = array_map('intval', array_filter(explode(',', $excluded_ids_raw)));
        if ( in_array($post_ID, $excluded_ids) ) {
            return;
        }

        $post_data = $this->get_single_post_data( $post );
        
        if ( $post_data ) {
            $message = sprintf(
                '¡Nuevo Post en el Blog! %s %s',
                $post_data['title'],
                $post_data['permalink']
            );

            $this->send_to_bluesky( $message, $post_data );
        }
    }
    
    /**
     * Obtiene los datos necesarios de un post para la publicación en Bluesky.
     * USANDO SÓLO FUNCIONES SEGURAS BASADAS EN OBJETO/ID para evitar fallos de CRON/setup_postdata.
     */
    private function get_single_post_data( $post ) {
        
        if ( ! $post ) {
            return null;
        }
        
        // 1. Obtención del título (Seguro: usa get_the_title(ID))
        $post_title = get_the_title( $post->ID ); 

        // 2. Obtención de permalink (Seguro: usa get_permalink(ID))
        $post_permalink = get_permalink( $post->ID ); 

        // 3. Obtención de extracto (Más seguro: usa get_post_field. Si está vacío, crea un extracto seguro del contenido)
        $raw_excerpt = get_post_field('post_excerpt', $post->ID);
        
        // Fallback: si no hay extracto (o está vacío), usar una parte del contenido.
        if ( empty(trim($raw_excerpt)) ) {
            $raw_excerpt = wp_strip_all_tags($post->post_content);
            $raw_excerpt = mb_substr($raw_excerpt, 0, 150); 
            $raw_excerpt .= '...';
        }
        $post_excerpt = $raw_excerpt;


        // 4. Obtención de fecha (Seguro: usa get_the_date(format, ID))
        $post_date = get_the_date( '', $post->ID ); 

        // 5. Obtención de imagen/thumbnail ID (Seguro: usa get_post_thumbnail_id(ID))
        $thumbnail_id = get_post_thumbnail_id($post->ID); 
        
        // 6. Obtención de URL y MIME 
        $image_url = $thumbnail_id ? wp_get_attachment_url($thumbnail_id) : null;
        $image_mime = $thumbnail_id ? get_post_mime_type($thumbnail_id) : null;
        
        $post_data = [
            'id'          => $post->ID,
            'years_ago'   => 0, 
            'title'       => $post_title,
            'permalink'   => $post_permalink,
            'date'        => $post_date,
            'excerpt'     => $post_excerpt,
            'image_url'   => $image_url,
            'image_mime'  => $image_mime,
        ];
        
        return $post_data;
    }

    public function enqueue_admin_assets( $hook ) {
        if ( 'toplevel_page_bsk-recordar' !== $hook ) {
            return;
        }

        $plugin_url = plugin_dir_url( __FILE__ ); 

        wp_enqueue_style( 'bsk-recordar-admin-style', $plugin_url . 'assets/css/bsk-recordar-admin.css', array(), '1.5.8' );

        wp_enqueue_script( 'jquery-ui-sortable' );
        wp_enqueue_script( 'bsk-recordar-admin-script', $plugin_url . 'assets/js/bsk-recordar-admin.js', array( 'jquery', 'jquery-ui-sortable' ), '1.5.8', true );
    }


    // 1. Crear el menú en el Admin
    public function add_admin_menu() {
        add_menu_page(
            'BSK Recordar', 
            'BSK Recordar', 
            'manage_options', 
            'bsk-recordar', 
            array( $this, 'render_admin_page' ), 
            'dashicons-calendar-alt', 
            100
        );
    }

    // 2. Registrar las opciones
    public function register_settings() {
        register_setting( 'bsk_recordar_group', 'bsk_handle' );
        register_setting( 'bsk_recordar_group', 'bsk_password' );
        register_setting( 'bsk_recordar_group', 'bsk_schedule_groups_array' ); 
        register_setting( 'bsk_recordar_group', 'bsk_autopublish_new' );
        register_setting( 'bsk_recordar_group', 'bsk_exclude_ids', array('sanitize_callback' => array($this, 'sanitize_exclude_ids')) );
        
        // Opciones de idioma
        register_setting( 'bsk_recordar_group', 'bsk_language_slug' );
        
        // Opcion para el Log de Diagnóstico (Oculta)
        register_setting( 'bsk_recordar_group', 'bsk_last_cron_status' );
        
        // Asegurar que la opción bsk_special_schedule_array se elimina/no se usa
        delete_option('bsk_special_schedule_array'); 
        delete_option('bsk_last_special_event_status');
    }

    /**
     * Limpia la cadena de IDs excluidas para asegurar que solo haya números y comas.
     */
    public function sanitize_exclude_ids($input) {
        $input = preg_replace('/[^0-9,]/', '', $input);
        return $input;
    }


    private function get_available_years() {
        $years = range(1, 20);
        $options = [];
        foreach ($years as $year) {
            $options[$year] = ($year == 1) ? 'Hace 1 año' : "Hace {$year} años";
        }
        return $options; 
    }

    // --- FUNCIÓN CON 30 MINUTOS DE INTERVALO ---
    private function get_available_times() {
        $times = [];
        for ($h = 0; $h < 24; $h++) {
            for ($m = 0; $m < 60; $m += 30) { 
                $time = sprintf('%02d:%02d:00', $h, $m);
                $times[$time] = date('H:i', strtotime($time)); 
            }
        }
        return $times;
    }

    /**
     * Obtiene los slugs y nombres de los idiomas disponibles si existe la taxonomía 'language'.
     */
    private function get_available_language_slugs() {
        if (!taxonomy_exists('language')) {
            return [];
        }
        $terms = get_terms([
            'taxonomy' => 'language',
            'hide_empty' => true,
        ]);
        $slugs = [];
        if (!is_wp_error($terms) && !empty($terms)) {
            foreach ($terms as $term) {
                $slugs[$term->slug] = $term->name . ' (' . $term->slug . ')';
            }
        }
        return $slugs;
    }


    // 3. Renderizar la página de configuración
    public function render_admin_page() {
        $message_status = '';
        $is_get_message = false;

        if ( isset( $_GET['bsk_message'] ) && $_GET['bsk_message'] === 'published' ) {
            $message_status = '<div class="notice notice-success is-dismissible"><p>La función de publicación se ha ejecutado. Revisa tu cuenta de Bluesky.</p></div>';
            $is_get_message = true;
        } elseif ( isset( $_GET['bsk_error'] ) ) {
            $error_message = sanitize_text_field(urldecode($_GET['bsk_error']));
            $message_status = '<div class="notice notice-error is-dismissible"><p>Error en la publicación manual: ' . $error_message . '</p></div>';
            $is_get_message = true;
        } elseif ( isset( $_GET['bsk_message'] ) && $_GET['bsk_message'] === 'cron_reset' ) {
            $message_status = '<div class="notice notice-success is-dismissible"><p>El sistema de CRON se ha **reiniciado** y reprogramado. Los posts de horas anteriores ya no se publicarán.</p></div>';
            $is_get_message = true;
        }
        
        // Lógica para enviar mensaje de prueba manual
        if ( isset( $_POST['bsk_test_message'] ) && check_admin_referer( 'bsk_send_test' ) ) {
            $test_message_text = stripslashes( $_POST['bsk_test_message'] );
            $result = $this->send_to_bluesky( $test_message_text );
            if ( is_wp_error( $result ) ) {
                $message_status = '<div class="notice notice-error is-dismissible"><p>Error: ' . $result->get_error_message() . '</p></div>';
            } else {
                $message_status = '<div class="notice notice-success is-dismissible"><p>¡Éxito! Post publicado en Bluesky.</p></div>';
            }
        }
        
        // --- INICIO LÓGICA PRUEBA HEMEROTECA ---
        $query_results_html = '';
        $test_date_value = date('Y-m-d'); 
        
        if ( isset( $_POST['bsk_test_query'] ) && check_admin_referer( 'bsk_test_query' ) ) {
            
            $test_date = isset($_POST['bsk_test_date']) ? sanitize_text_field($_POST['bsk_test_date']) : null;
            $test_date_value = $test_date ? $test_date : date('Y-m-d'); 
            
            // Usar un rango amplio de años (1 a 20) para la PRUEBA DE DIAGNÓSTICO
            $years_to_check = range(1, 20); 
            
            $posts_found = $this->bsk_recordar_query_posts( $test_date, $years_to_check, [] );
            
            if ( ! empty( $posts_found ) ) {
                $tax_msg = taxonomy_exists('language') ? 'según el filtro de idioma' : 'sin filtrar por idioma';
                $query_results_html = '<div class="notice notice-info is-dismissible"><h3>Resultados de la Búsqueda de Hemeroteca para ' . date('d/m/Y', strtotime($test_date_value)) . ' ' . $tax_msg . ':</h3><ul>';
                foreach ($posts_found as $post_data) {
                    $has_image = ! empty($post_data['image_url']) ? '✅' : '❌';
                    $plural = ($post_data['years_ago'] == 1) ? 'año' : 'años';
                    $query_results_html .= '<li><strong>ID ' . $post_data['id'] . '</strong>: Hace ' . $post_data['years_ago'] . ' ' . $plural . ': ' . esc_html($post_data['title']) . ' (Imagen: ' . $has_image . ')</li>';
                }
                $query_results_html .= '</ul><p>Nota: Los posts excluidos por ID no aparecen en esta lista si la ID es correcta.</p></div>';
            } else {
                $tax_msg_detail = taxonomy_exists('language') ? ' (según el filtro de idioma actual)' : '';
                $query_results_html = '<div class="notice notice-warning is-dismissible"><h3>Resultados de la Búsqueda de Hemeroteca:</h3><p>No se encontraron posts para las efemérides de ' . date('d/m/Y', strtotime($test_date_value)) . ' en el rango de 1 a 20 años de antigüedad' . $tax_msg_detail . '.</p></div>';
            }
        }
        // --- FIN LÓGICA PRUEBA HEMEROTECA ---
        
        $current_schedule = json_decode(get_option('bsk_schedule_groups_array', '[]'), true);
        $available_years_options = $this->get_available_years();
        $available_times = $this->get_available_times(); 
        $excluded_ids_value = get_option('bsk_exclude_ids', '');
        
        $language_options = $this->get_available_language_slugs();
        $current_lang_slug = get_option('bsk_language_slug', 'es'); 
        
        // V1.5.8: DIAGNOSTIC DATA
        $last_cron_status = get_option('bsk_last_cron_status', 'N/A (Aún no se ha ejecutado el CRON)');

        ?>
        <div class="wrap">
            
            <h1>Configuración BSK Recordar (v1.5.8-CLEAN-STABLE)</h1>
            <?php echo $message_status; ?>

            <?php
            // --- INICIO SCRIPT DE LIMPIEZA DE URL ---
            if ($is_get_message) {
                $current_url = admin_url( 'admin.php?page=bsk-recordar' );
                $message_keys = array('bsk_message', 'bsk_error');
                $clean_url = remove_query_arg($message_keys, $current_url);
                
                echo '<script type="text/javascript">
                    if (typeof history.replaceState !== "undefined") {
                        history.replaceState(null, null, "' . esc_url($clean_url) . '");
                    }
                </script>';
            }
            // --- FIN SCRIPT DE LIMPIEZA DE URL ---
            ?>
            
            <form id="bsk-recordar-form" method="post" action="options.php">
                <?php settings_fields( 'bsk_recordar_group' ); ?>
                <?php do_settings_sections( 'bsk_recordar_group' ); ?>
                
                <h2>Credenciales de Bluesky</h2>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">Bluesky Handle</th>
                        <td><input type="text" name="bsk_handle" value="<?php echo esc_attr( get_option('bsk_handle') ); ?>" class="regular-text" placeholder="ejemplo.bsky.social" /></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">App Password</th>
                        <td><input type="password" name="bsk_password" value="<?php echo esc_attr( get_option('bsk_password') ); ?>" class="regular-text" /></td>
                    </tr>
                </table>

                <h2>Control Editorial y Automatización</h2>
                <table class="form-table">
                    
                    <?php if (taxonomy_exists('language')): ?>
                    <tr valign="top">
                        <th scope="row">Idioma de Publicación (Efemérides)</th>
                        <td>
                            <select name="bsk_language_slug">
                                <option value="all" <?php selected($current_lang_slug, 'all'); ?>>Todos los idiomas (Sin filtrar)</option>
                                <option value="none" <?php selected($current_lang_slug, 'none'); ?>>-- No publicar ningún idioma (Desactivado) --</option>
                                <option disabled>──────────</option>
                                <?php foreach ($language_options as $slug => $name): ?>
                                    <option value="<?php echo esc_attr($slug); ?>" <?php selected($current_lang_slug, $slug); ?>>
                                        <?php echo esc_html($name); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description">Selecciona el idioma de los posts que se deben publicar como efemérides. **Solo visible si usas un plugin de idioma**.</p>
                        </td>
                    </tr>
                    <?php endif; ?>
                    
                    <tr valign="top">
                        <th scope="row">Excluir Posts por ID</th>
                        <td>
                            <input type="text" name="bsk_exclude_ids" value="<?php echo esc_attr( $excluded_ids_value ); ?>" class="large-text" placeholder="Ej: 12, 345, 6789" />
                            <p class="description">Introduce los IDs de los posts de WordPress que **NUNCA** deben ser republicados por la hemeroteca ni publicados inmediatamente. Separa los números con comas.</p>
                        </td>
                    </tr>
                    
                    <tr valign="top">
                        <th scope="row">Publicar Posts Nuevos Inmediatamente</th>
                        <td>
                            <?php $autopublish_new = get_option('bsk_autopublish_new', 'off'); ?>
                            <input type="checkbox" name="bsk_autopublish_new" value="on" <?php checked( $autopublish_new, 'on' ); ?> /> 
                            <p class="description">Si está activado, se enviará inmediatamente a Bluesky cada vez que publiques un post nuevo (a menos que su ID esté excluida).</p>
                        </td>
                    </tr>
                </table>

                <h3>Programación de Efemérides (General)</h3>
                <p class="description">Define cuándo se buscarán y publicarán posts antiguos ("Hace N años"). (Intervalos de 30 minutos)</p>
                <table class="widefat fixed" id="bsk-schedule-table">
                    <thead>
                        <tr>
                            <th style="width: 40%;">Años de Antigüedad (Hace N años)</th>
                            <th style="width: 30%;">Hora de Publicación (Servidor)</th>
                            <th style="width: 10%;">Acción</th>
                        </tr>
                    </thead>
                    <tbody id="bsk-schedule-rows">
                        <?php 
                        // Renderizar las filas existentes si las hay
                        if (!empty($current_schedule)): ?>
                            <?php foreach ($current_schedule as $item): ?>
                                <tr class="bsk-schedule-row">
                                    <td>
                                        <select class="bsk-year-selector" required>
                                            <?php foreach ($available_years_options as $year_value => $year_label): ?>
                                                <option value="<?php echo $year_value; ?>" <?php selected($item['year'], $year_value); ?>><?php echo $year_label; ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <select class="bsk-time-selector" required>
                                            <?php foreach ($available_times as $value => $label): ?>
                                                <option value="<?php echo $value; ?>" <?php selected($item['time'], $value); ?>>
                                                    <?php echo $label; ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td><button type="button" class="button button-secondary bsk-remove-row">Eliminar</button></td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
                <p style="margin-top: 10px;">
                    <button type="button" id="bsk-add-row" class="button button-primary">Añadir Horario</button>
                </p>
                <input type="hidden" name="bsk_schedule_groups_array" id="bsk-schedule-groups-array" value="<?php echo esc_attr(get_option('bsk_schedule_groups_array', '[]')); ?>" />
                
                <?php submit_button('Guardar Cambios de Configuración'); ?>
            </form>
            
            <hr>

            <h2>Diagnóstico de CRON</h2>
            <div style="background: #f9f9e9; border: 1px solid #ddd; padding: 10px; margin-top: 20px;">
                <p><strong>Último Estado del CRON (Tras la última hora programada):</strong></p>
                <p><code><?php echo esc_html($last_cron_status); ?></code></p>
                <p class="description">Este mensaje se actualiza después de la ejecución de cualquier CRON programado. Ayuda a diagnosticar fallos de *Missed Event* o si la búsqueda fue exitosa. (Formato: `CRON ejecutado a [HORA] el [FECHA] (Hora Compensada): [RESULTADO]`) </p>
            </div>
            
            <hr>

            <h2>Prueba de Conexión (Publicación de Texto Plano)</h2>
            <form method="post" action="">
                <?php wp_nonce_field( 'bsk_send_test' ); ?>
                <textarea name="bsk_test_message" rows="3" class="large-text" placeholder="Escribe algo para probar..."></textarea>
                <br><br>
                <input type="submit" name="bsk_test_publish" class="button button-secondary" value="Enviar Post de Prueba a Bluesky">
            </form>

            <hr>
            
            <?php echo $query_results_html; ?>

            <h2>Prueba de Hemeroteca (Consulta General)</h2>
            <p>Prueba la lógica de búsqueda de efemérides (no publica nada) en el rango de los últimos 20 años, teniendo en cuenta las IDs excluidas.</p>
            <form method="post" action="">
                <?php wp_nonce_field( 'bsk_test_query' ); ?>
                <label for="bsk_test_date">Fecha de Prueba:</label>
                <input type="date" id="bsk_test_date" name="bsk_test_date" value="<?php echo esc_attr($test_date_value); ?>" class="regular-text" />
                <br><br>
                <input type="submit" name="bsk_test_query" class="button button-secondary" value="Buscar Efemérides en la Fecha Seleccionada">
            </form>
            
            <hr>

            <h2>Prueba de Automatización (Publicación Manual de Efemérides)</h2>
            <p>La publicación manual solo funcionará con los años que estén **guardados** en la tabla de *Programación de Efemérides (General)*.</p>
            <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
                <input type="hidden" name="action" value="bsk_test_publish" />
                <?php wp_nonce_field( 'bsk_manual_publish' ); ?>
                
                <label for="bsk_manual_publish_date">Fecha para la prueba de publicación:</label>
                <input type="date" id="bsk_manual_publish_date" name="bsk_manual_publish_date" value="<?php echo date('Y-m-d'); ?>" class="regular-text" /> 
                <br><br>
                <input type="submit" class="button button-primary" value="Publicar Efemérides de la Fecha Seleccionada (Test todos los Grupos Guardados)">
            </form>

            <hr>
            <div style="background: #eef; border: 1px solid #cce; padding: 15px; margin-top: 20px;">
                <h2>⚙️ Herramientas de Mantenimiento</h2>
                
                <p>Esta herramienta es para diagnóstico y solución de problemas, especialmente si el sistema de publicación se detiene o intenta publicar efemérides de horas pasadas.</p>

                <h3>Reiniciar Programación CRON</h3>
                <p>Úsalo si detectas comportamientos erráticos de CRON.</p>
                <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>" style="display: inline-block;">
                    <input type="hidden" name="action" value="bsk_cron_reset" />
                    <?php wp_nonce_field( 'bsk_cron_reset' ); ?>
                    <input type="submit" class="button button-secondary" value="FORZAR REINICIO DEL CRON" />
                </form>
            </div>
            </div>
        <?php
        
        // ----------------------------------------------------
        // TEMPLATES OCULTOS PARA JAVASCRIPT
        // ----------------------------------------------------

        $years_options = '';
        foreach ($available_years_options as $year_value => $year_label) {
            $years_options .= "<option value='{$year_value}'>{$year_label}</option>";
        }

        $times_options = ''; 
        foreach ($available_times as $value => $label) {
            $times_options .= "<option value='{$value}'>{$label}</option>";
        }
        ?>
        <template id="bsk-row-template">
            <tr class="bsk-schedule-row">
                <td>
                    <select class="bsk-year-selector" required>
                        <?php echo $years_options; ?>
                    </select>
                </td>
                <td>
                    <select class="bsk-time-selector" required>
                        <?php echo $times_options; ?>
                    </select>
                </td>
                <td><button type="button" class="button button-secondary bsk-remove-row">Eliminar</button></td>
            </tr>
        </template>
        <?php
    }

    private function upload_image_to_bluesky( $token, $image_url, $mime_type ) {
        $image_response = wp_remote_get( $image_url );
        if ( is_wp_error( $image_response ) ) return $image_response;
        $image_data = wp_remote_retrieve_body( $image_response );
        $upload_response = wp_remote_post( $this->api_base . 'com.atproto.repo.uploadBlob', array(
            'headers' => array('Authorization' => 'Bearer ' . $token, 'Content-Type'  => $mime_type),
            'body' => $image_data,
        ));
        if ( is_wp_error( $upload_response ) ) return $upload_response;
        
        $response_code = wp_remote_retrieve_response_code( $upload_response );
        $response_body = wp_remote_retrieve_body( $upload_response );

        if ( $response_code !== 200 ) {
            $error_message = json_decode($response_body, true)['message'] ?? 'Error desconocido al subir blob.';
            return new WP_Error( 'upload_failed_http', "Error HTTP $response_code al subir la imagen. Detalles: $error_message" );
        }

        $upload_body = json_decode( $response_body, true );
        if ( ! isset( $upload_body['blob'] ) ) {
            return new WP_Error( 'upload_failed', 'Error al subir la imagen: No se encontró blob en la respuesta.' );
        }
        return $upload_body['blob'];
    }

    public function send_to_bluesky( $text, $post_data = [] ) {
        $handle = get_option( 'bsk_handle' ); 
        $password = get_option( 'bsk_password' );

        if ( ! $handle || ! $password ) { 
            return new WP_Error( 'missing_creds', 'Faltan credenciales. Por favor, configura el Handle y la App Password.' ); 
        }
        $password_length = strlen($password);
        
        // --- 1. Create Session (Autenticación) ---
        $session_response = wp_remote_post( $this->api_base . 'com.atproto.server.createSession', array(
            'headers' => array( 'Content-Type' => 'application/json' ),
            'body'    => json_encode( array('identifier' => $handle, 'password' => $password)),
        ));
        
        if ( is_wp_error( $session_response ) ) return $session_response;
        
        $response_body = wp_remote_retrieve_body( $session_response );
        $response_code = wp_remote_retrieve_response_code( $session_response );
        $session_body = json_decode( $response_body, true );
        
        if ( $response_code !== 200 ) {
            $error_detail = (isset($session_body['message']) && is_string($session_body['message'])) ? $session_body['message'] : 'Respuesta no JSON o error desconocido.';
            
            return new WP_Error( 
                'auth_failed_http', 
                "Autenticación fallida (HTTP $response_code). Verifique Handle/App Password. Detalles: $error_detail. (DEBUG: Longitud App Pass: {$password_length})" 
            );
        }

        if ( ! isset( $session_body['accessJwt'] ) ) { 
            return new WP_Error( 'auth_failed_token', 'Error al obtener el token de acceso. Respuesta: ' . $response_body ); 
        }

        $access_token = $session_body['accessJwt']; 
        $did = $session_body['did'];
        $embed = null;

        // --- 2. Preparar Embed y Posible Imagen ---
        if ( ! empty( $post_data['permalink'] ) ) {
            $image_blob = null;
            if ( ! empty( $post_data['image_url'] ) ) { 
                $image_blob = $this->upload_image_to_bluesky( $access_token, $post_data['image_url'], $post_data['image_mime'] ); 
                if ( is_wp_error( $image_blob ) ) return $image_blob;
            }
            
            $raw_description = ! empty($post_data['excerpt']) ? $post_data['excerpt'] : $post_data['title'];
            $clean_description = html_entity_decode(wp_strip_all_tags($raw_description), ENT_QUOTES, 'UTF-8');
            
            $embed_external = array(
                '$type' => 'app.bsky.embed.external',
                'external' => array(
                    'uri' => $post_data['permalink'], 'title' => $post_data['title'],
                    'description' => $clean_description, 
                )
            );
            if ( ! empty( $image_blob ) ) { $embed_external['external']['thumb'] = $image_blob; }
            $embed = $embed_external;
        }

        $facets = [];
        if ( ! empty( $post_data['permalink'] ) ) {
            $permalink = $post_data['permalink']; $link_start_pos = strrpos($text, $permalink);
            if ($link_start_pos !== false) {
                $link_start_byte = $link_start_pos; $link_end_byte = $link_start_pos + strlen($permalink);
                $facets = [[
                    '$type' => 'app.bsky.richtext.facet',
                    'index' => ['byteStart' => $link_start_byte, 'byteEnd'   => $link_end_byte],
                    'features' => [[ '$type' => 'app.bsky.richtext.facet#link', 'uri'   => $permalink]]
                ]];
            }
        }
        $now = date( 'c' ); 
        $record = array('$type' => 'app.bsky.feed.post', 'text' => $text, 'createdAt' => $now);
        if (!empty($facets)) { $record['facets'] = $facets; }
        if ($embed) { $record['embed'] = $embed; }
        $post_body = array('repo' => $did, 'collection' => 'app.bsky.feed.post', 'record' => $record);
        
        // --- 3. Create Post ---
        $post_response = wp_remote_post( $this->api_base . 'com.atproto.repo.createRecord', array(
            'headers' => array('Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $access_token),
            'body' => json_encode( $post_body ),
        ));
        
        if ( is_wp_error( $post_response ) ) return $post_response;
        
        $post_response_code = wp_remote_retrieve_response_code( $post_response );
        $post_response_body = wp_remote_retrieve_body( $post_response );
        
        if ( $post_response_code !== 200 ) { 
            $error_message = json_decode($post_response_body, true)['message'] ?? 'Error desconocido al publicar.';
            return new WP_Error( 'post_failed', "Error HTTP $post_response_code al publicar. Detalles: $error_message" );
        }
        
        return true;
    }


    /**
     * Consulta los posts de la hemeroteca.
     */
    public function bsk_recordar_query_posts( $test_date = null, $years_to_check = [], $extra_excluded_ids = [] ) {
        if (empty($years_to_check)) {
            return [];
        }
        
        // 1. OBTENER Y PREPARAR IDS EXCLUIDAS
        $excluded_ids_raw = get_option('bsk_exclude_ids', '');
        $excluded_ids = array_map('intval', array_filter(explode(',', $excluded_ids_raw)));
        
        // El tercer argumento ya no es necesario, pero lo mantenemos para compatibilidad con la llamada si existe
        $excluded_ids = array_merge($excluded_ids, $extra_excluded_ids);
        if ( empty($excluded_ids) ) {
            $excluded_ids = array(0);
        }

        $timestamp = $test_date ? strtotime($test_date) : time();
        $today_day = date( 'd', $timestamp ); 
        $today_month = date( 'm', $timestamp ); 
        $current_year = date( 'Y', $timestamp );
        $posts_to_publish = [];

        if ( ! function_exists( 'is_leap_year' ) ) {
            function is_leap_year( $year ) {
                return ( ( $year % 4 == 0 ) && ( $year % 100 != 0 ) ) || ( $year % 400 == 0 );
            }
        }

        // ------------------------------------------------------------------
        // DETECCIÓN Y FILTRO DE IDIOMA
        // ------------------------------------------------------------------
        $tax_query = array();

        if (taxonomy_exists('language')) {
            $language_slug = get_option('bsk_language_slug', 'es');

            if ($language_slug === 'none') {
                return []; 
            }

            if ($language_slug !== 'all') {
                 $tax_query = array(
                    array(
                        'taxonomy' => 'language',
                        'field' => 'slug',
                        'terms' => $language_slug
                    )
                 );
            }
        }
        // ------------------------------------------------------------------

        foreach ( $years_to_check as $years_ago ) {
            if ($years_ago <= 0) continue;
            $target_year = $current_year - $years_ago;
            
            if ( $today_day === '29' && $today_month === '02' && ! is_leap_year( $target_year ) ) {
                continue; 
            }
            $args = array(
                'post_type' => 'post', 'post_status' => 'publish',
                'posts_per_page' => 1, 'orderby' => 'date', 'order' => 'DESC', 
                'date_query' => array(
                    'relation' => 'AND',
                    array( 'year' => $target_year, 'month' => $today_month, 'day' => $today_day),
                ),
                'post__not_in' => array_filter(array_unique($excluded_ids)),
                'no_found_rows' => true, 
            );

            if ( ! empty( $tax_query ) ) {
                $args['tax_query'] = $tax_query;
            }

            $query = new WP_Query( $args );

            if ( $query->have_posts() ) {
                while ( $query->have_posts() ) {
                    $query->the_post();
                    global $post;
                    
                    $post_data = $this->get_single_post_data( $post );
                    
                    if ( $post_data ) {
                        $post_data['years_ago'] = $years_ago;
                        $posts_to_publish[] = $post_data;
                    }
                }
                wp_reset_postdata();
            }
        }
        return $posts_to_publish;
    }
}

new BSK_Recordar();